﻿using _25_HasArelationship._2_BankaHesaplari.Config;
using _25_HasArelationship._2_BankaHesaplari.Enums;

namespace _25_HasArelationship._2_BankaHesaplari.Bases
{
    /// <summary>
    /// Bireysel ve kurumsal hesap somut sınıflarının miras alacağı soyut sınıf.
    /// </summary>
    public abstract class HesapBase
    {
        #region Özellikler
        /// <summary>
        /// Tekil otomatik bir artacak hesap numarası.
        /// </summary>
        public int Id { get; set; }

        /// <summary>
        /// Hesap sahibinin adı ve soyadı.
        /// </summary>
        public string Sahibi { get; set; }

        /// <summary>
        /// Hesabın aktif hesap veya aktif olmayan (dondurulmuş) hesap değeri.
        /// </summary>
        public bool AktifMi { get; set; } = true; // ilk değer ataması olarak true yapıyoruz ki yeni hesap objesi oluşturulduğunda hesap aktif olsun

        /// <summary>
        /// Hesaptaki işlemler üzerinden sadece okunur toplam para miktarı (bakiye).
        /// </summary>
        public decimal ToplamMiktar
        {
            get
            {
                decimal toplam = 0;
                foreach (Islem islem in _islemler)
                {
                    toplam += islem.Miktar;
                }
                return toplam;
            }
        }

        /// <summary>
        /// Hesaptaki toplam para miktarına faiz oranı uygulanmış miktar.
        /// </summary>
        public decimal FaizliToplamMiktar
        {
            get
            {
                return ToplamMiktar + ToplamMiktar * _faizOrani;
            }
        }
        #endregion



        #region Alanlar
        /// <summary>
        /// Bakiyeye uygulanacak faiz oranı.
        /// </summary>
        protected decimal _faizOrani; // protected tanımladık ki bireysel ve kurumsal hesap sınıflarında erişebilelim

        /// <summary>
        /// Hesabın para yatırma ve para çekme işlemleri.
        /// </summary>
        private List<IslemBase> _islemler = new List<IslemBase>(); // has-a relationship: bir hesabın sıfır veya daha çok işlemi olabilir,
                                                                   // hesap objesi oluştuğunda işlemler koleksiyonu da oluşacak
        #endregion



        #region Constructor
        protected HesapBase() // class abstract olduğu için constructor'ı protected tanımlıyoruz
        {
            Id = HesapConfig.Id++; // tüm uygulama için ortak paylaşılan statik HesapConfig sınıfındaki Id'nin değerini hesabın Id'sine ata ve 1 arttır
        }
        #endregion



        #region Davranışlar
        /// <summary>
        /// Hesaba para yatırma işlemi.
        /// </summary>
        /// <param name="yatirilacakMiktar"></param>
        /// <returns>IslemSonucu</returns>
        public IslemSonucu ParaYatir(decimal yatirilacakMiktar)
        {
            if (yatirilacakMiktar <= 0) // yatırılacak miktar 0'dan küçük veya 0'a eşitse
                return IslemSonucu.NegatifMiktar; // hatalı işlem sonucu dönüyoruz

            // yeni bir işlem objesinin oluşturulup işlem listesine eklenmesi
            Islem islem = new Islem(yatirilacakMiktar, true);
            _islemler.Add(islem);

            return IslemSonucu.Basarili; // başarılı işlem sonucu dönüyoruz
        }

        /// <summary>
        /// Hesaptan para çekme işlemi.
        /// </summary>
        /// <param name="cekilecekMiktar"></param>
        /// <returns>IslemSonucu</returns>
        public IslemSonucu ParaCek(decimal cekilecekMiktar)
        {
            if (cekilecekMiktar <= 0) // çekilecek miktar 0'dan küçük veya 0'a eşitse
                return IslemSonucu.NegatifMiktar; // hatalı işlem sonucu dönüyoruz

            if (ToplamMiktar - cekilecekMiktar < 0) // eğer para çekildikten sonra bakiye negatifse
                return IslemSonucu.YetersizBakiye; // hatalı işlem sonucu dönüyoruz

            // yeni bir işlem objesinin oluşturulup işlem listesine eklenmesi
            Islem islem = new Islem(-cekilecekMiktar, false); // -cekilecekMiktar = -1 * cekilecekMiktar
            _islemler.Add(islem);

            return IslemSonucu.Basarili; // başarılı işlem sonucu dönüyoruz
        }

        /// <summary>
        /// Bu sınıftan miras alan bireysel ve kurumsal hesap sınıflarında farklı oranlar üzerinden toplam miktara 
        /// uygulanacak faiz oranını atayan soyut method tanımı.
        /// </summary>
        public abstract void FaizOraniUygula();
        #endregion
    }
}
