﻿using Locations.APP.Domain;
using Microsoft.AspNetCore.Mvc;
using Microsoft.EntityFrameworkCore;

namespace Locations.API.Controllers
{
    /// <summary>
    /// API controller for database management operations such as seeding initial data.
    /// </summary>
    [Route("api/[controller]")]
    [ApiController]
    public class DatabaseController : ControllerBase
    {
        /// <summary>
        /// The database context for accessing and manipulating country and city data.
        /// </summary>
        private readonly LocationsDb _db;

        /// <summary>
        /// Provides information about the web hosting environment (e.g., Development, Production).
        /// </summary>
        private readonly IWebHostEnvironment _environment;

        /// <summary>
        /// Initializes a new instance of the <see cref="DatabaseController"/> class.
        /// </summary>
        /// <param name="db">The database context to be used for data operations.</param>
        /// <param name="environment">The web hosting environment.</param>
        public DatabaseController(LocationsDb db, IWebHostEnvironment environment)
        {
            _db = db;
            _environment = environment;
        }

        /// <summary>
        /// Seeds the database with initial country and city data for development purposes.
        /// This action removes all existing countries and cities, resets identity columns, and inserts predefined data for Türkiye and the United States of America.
        /// </summary>
        /// <remarks>
        /// This endpoint is only available in the development environment to prevent accidental data loss in production.
        /// </remarks>
        /// <returns>
        /// Returns HTTP 200 OK with a success message if seeding is successful; otherwise, returns HTTP 400 Bad Request if not in development.
        /// </returns>
        [HttpGet, Route("~/api/SeedDb")]
        public IActionResult Seed()
        {
            // Can be uncommented to ensure that the seed operation is only allowed in the development environment.
            //if (!_environment.IsDevelopment())
            //    return BadRequest("The seed operation can only be performed in development environment!");

            // Remove all existing city and country records from the database.
            _db.Cities.RemoveRange(_db.Cities.ToList());
            _db.Countries.RemoveRange(_db.Countries.ToList());

            // Reset the auto-increment (identity) columns for Cities and Countries tables.
            _db.Database.ExecuteSqlRaw("UPDATE SQLITE_SEQUENCE SET SEQ=0 WHERE NAME='Cities';");
            _db.Database.ExecuteSqlRaw("UPDATE SQLITE_SEQUENCE SET SEQ=0 WHERE NAME='Countries';");

            // Add a new Country entity for Türkiye, including all 81 cities as child entities.
            _db.Countries.Add(new Country
            {
                CountryName = "Türkiye",
                Cities = new List<City>
                {
                    new City { CityName = "Adana" },
                    new City { CityName = "Adıyaman" },
                    new City { CityName = "Afyonkarahisar" },
                    new City { CityName = "Ağrı" },
                    new City { CityName = "Amasya" },
                    new City { CityName = "Ankara" },
                    new City { CityName = "Antalya" },
                    new City { CityName = "Artvin" },
                    new City { CityName = "Aydın" },
                    new City { CityName = "Balıkesir" },
                    new City { CityName = "Bilecik" },
                    new City { CityName = "Bingöl" },
                    new City { CityName = "Bitlis" },
                    new City { CityName = "Bolu" },
                    new City { CityName = "Burdur" },
                    new City { CityName = "Bursa" },
                    new City { CityName = "Çanakkale" },
                    new City { CityName = "Çankırı" },
                    new City { CityName = "Çorum" },
                    new City { CityName = "Denizli" },
                    new City { CityName = "Diyarbakır" },
                    new City { CityName = "Edirne" },
                    new City { CityName = "Elazığ" },
                    new City { CityName = "Erzincan" },
                    new City { CityName = "Erzurum" },
                    new City { CityName = "Eskişehir" },
                    new City { CityName = "Gaziantep" },
                    new City { CityName = "Giresun" },
                    new City { CityName = "Gümüşhane" },
                    new City { CityName = "Hakkari" },
                    new City { CityName = "Hatay" },
                    new City { CityName = "Isparta" },
                    new City { CityName = "Mersin" },
                    new City { CityName = "İstanbul" },
                    new City { CityName = "İzmir" },
                    new City { CityName = "Kars" },
                    new City { CityName = "Kastamonu" },
                    new City { CityName = "Kayseri" },
                    new City { CityName = "Kırklareli" },
                    new City { CityName = "Kırşehir" },
                    new City { CityName = "Kocaeli" },
                    new City { CityName = "Konya" },
                    new City { CityName = "Kütahya" },
                    new City { CityName = "Malatya" },
                    new City { CityName = "Manisa" },
                    new City { CityName = "Kahramanmaraş" },
                    new City { CityName = "Mardin" },
                    new City { CityName = "Muğla" },
                    new City { CityName = "Muş" },
                    new City { CityName = "Nevşehir" },
                    new City { CityName = "Niğde" },
                    new City { CityName = "Ordu" },
                    new City { CityName = "Rize" },
                    new City { CityName = "Sakarya" },
                    new City { CityName = "Samsun" },
                    new City { CityName = "Siirt" },
                    new City { CityName = "Sinop" },
                    new City { CityName = "Sivas" },
                    new City { CityName = "Tekirdağ" },
                    new City { CityName = "Tokat" },
                    new City { CityName = "Trabzon" },
                    new City { CityName = "Tunceli" },
                    new City { CityName = "Şanlıurfa" },
                    new City { CityName = "Uşak" },
                    new City { CityName = "Van" },
                    new City { CityName = "Yozgat" },
                    new City { CityName = "Zonguldak" },
                    new City { CityName = "Aksaray" },
                    new City { CityName = "Bayburt" },
                    new City { CityName = "Karaman" },
                    new City { CityName = "Kırıkkale" },
                    new City { CityName = "Batman" },
                    new City { CityName = "Şırnak" },
                    new City { CityName = "Bartın" },
                    new City { CityName = "Ardahan" },
                    new City { CityName = "Iğdır" },
                    new City { CityName = "Yalova" },
                    new City { CityName = "Karabük" },
                    new City { CityName = "Kilis" },
                    new City { CityName = "Osmaniye" },
                    new City { CityName = "Düzce" }
                }
            });

            // Add a new Country entity for the United States of America, including a representative list of major cities.
            _db.Countries.Add(new Country
            {
                CountryName = "United States of America",
                Cities = new List<City>
                {
                    new City { CityName = "New York" },
                    new City { CityName = "Los Angeles" },
                    new City { CityName = "Chicago" },
                    new City { CityName = "Houston" },
                    new City { CityName = "Phoenix" },
                    new City { CityName = "Philadelphia" },
                    new City { CityName = "San Antonio" },
                    new City { CityName = "San Diego" },
                    new City { CityName = "Dallas" },
                    new City { CityName = "San Jose" },
                    new City { CityName = "Austin" },
                    new City { CityName = "Jacksonville" },
                    new City { CityName = "Fort Worth" },
                    new City { CityName = "Columbus" },
                    new City { CityName = "Charlotte" },
                    new City { CityName = "San Francisco" },
                    new City { CityName = "Indianapolis" },
                    new City { CityName = "Seattle" },
                    new City { CityName = "Denver" },
                    new City { CityName = "Washington" },
                    new City { CityName = "Boston" },
                    new City { CityName = "El Paso" },
                    new City { CityName = "Nashville" },
                    new City { CityName = "Detroit" },
                    new City { CityName = "Oklahoma City" },
                    new City { CityName = "Portland" },
                    new City { CityName = "Las Vegas" },
                    new City { CityName = "Memphis" },
                    new City { CityName = "Louisville" },
                    new City { CityName = "Baltimore" },
                    new City { CityName = "Milwaukee" },
                    new City { CityName = "Albuquerque" },
                    new City { CityName = "Tucson" },
                    new City { CityName = "Fresno" },
                    new City { CityName = "Mesa" },
                    new City { CityName = "Sacramento" },
                    new City { CityName = "Atlanta" },
                    new City { CityName = "Kansas City" },
                    new City { CityName = "Colorado Springs" },
                    new City { CityName = "Miami" },
                    new City { CityName = "Raleigh" },
                    new City { CityName = "Omaha" },
                    new City { CityName = "Long Beach" },
                    new City { CityName = "Virginia Beach" },
                    new City { CityName = "Oakland" },
                    new City { CityName = "Minneapolis" },
                    new City { CityName = "Tulsa" },
                    new City { CityName = "Arlington" },
                    new City { CityName = "New Orleans" }
                }
            });

            // Add a new Country entity for China with no cities.
            _db.Countries.Add(new Country
            {
                CountryName = "China",
            });

            // Persist all changes to the database.
            _db.SaveChanges();

            // Return a success message indicating the database was seeded.
            return Ok("Database seed successful.");
        }
    }
}
